"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = __importDefault(require("express"));
const User_1 = __importDefault(require("../models/User"));
const bcrypt_1 = __importDefault(require("bcrypt"));
const Subscription_1 = __importDefault(require("../models/Subscription"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const dotenv_1 = __importDefault(require("dotenv"));
const uploadMiddleware_1 = __importDefault(require("../middleware/uploadMiddleware"));
const asyncHandler_1 = require("../middleware/asyncHandler");
const passport_1 = __importDefault(require("passport"));
const passport_google_oauth20_1 = require("passport-google-oauth20");
const passport_facebook_1 = require("passport-facebook");
const isAuth_1 = __importDefault(require("../middleware/isAuth"));
const Organization_1 = __importDefault(require("../models/Organization"));
dotenv_1.default.config();
const router = express_1.default.Router();
const generateToken = (userId) => {
    const payload = { userId };
    const secretKey = process.env.JWT_SECRET || '2a7e72426f642b756c44397728';
    const options = { expiresIn: '5Hrs' };
    return jsonwebtoken_1.default.sign(payload, secretKey, options);
};
passport_1.default.serializeUser((user, done) => done(null, user.id));
passport_1.default.deserializeUser((id, done) => done(null, id));
// Configure Passport for Google OAuth
passport_1.default.use(new passport_google_oauth20_1.Strategy({
    clientID: process.env.GOOGLE_CLIENT_ID || '',
    clientSecret: process.env.GOOGLE_CLIENT_SECRET || '',
    callbackURL: `${process.env.API_URL || 'https://api.voicecon.ai:5000'}/api/v1/auth/google/callback`,
    passReqToCallback: true
}, (req, accessToken, refreshToken, profile, done) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        let user = yield User_1.default.findOne({
            where: { email: profile.emails[0].value },
            include: [{ model: Subscription_1.default, as: 'subscriptions' }]
        });
        if (!user) {
            user = yield User_1.default.create({
                email: profile.emails[0].value,
                name: profile.displayName,
                password: yield bcrypt_1.default.hash(Math.random().toString(36).slice(-8), 10), // random password
                profile_url: profile.photos && profile.photos[0] ? profile.photos[0].value : null
            });
        }
        return done(null, user);
    }
    catch (error) {
        return done(error);
    }
})));
// Configure Passport for Facebook OAuth
passport_1.default.use(new passport_facebook_1.Strategy({
    clientID: process.env.FACEBOOK_APP_ID || '',
    clientSecret: process.env.FACEBOOK_APP_SECRET || '',
    callbackURL: `${process.env.API_URL || 'https://api.voicecon.ai:5000'}/api/v1/auth/facebook/callback`,
    profileFields: ['id', 'emails', 'name', 'picture.type(large)'],
    passReqToCallback: true
}, (req, accessToken, refreshToken, profile, done) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Check if user exists
        let user = yield User_1.default.findOne({
            where: { email: profile.emails[0].value },
            include: [{ model: Subscription_1.default, as: 'subscriptions' }]
        });
        if (!user) {
            // Create new user if doesn't exist
            user = yield User_1.default.create({
                email: profile.emails[0].value,
                name: `${profile.name.givenName} ${profile.name.familyName}`,
                password: yield bcrypt_1.default.hash(Math.random().toString(36).slice(-8), 10), // Generate random password
                profile_url: profile.photos && profile.photos[0] ? profile.photos[0].value : null
            });
        }
        return done(null, user);
    }
    catch (error) {
        return done(error);
    }
})));
// Passport initialization
router.use(passport_1.default.initialize());
// Google OAuth routes
router.get('/google', passport_1.default.authenticate('google', {
    scope: ['profile', 'email']
}));
router.get('/google/callback', passport_1.default.authenticate('google', {
    session: false,
    failureRedirect: process.env.FRONTEND_URL + '/login'
}), (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const userId = user.id.toString();
        const token = generateToken(userId);
        // Format user data similar to regular login
        // const userData = {
        //   id: user.id,
        //   email: user.email,
        //   name: user.name,
        //   profile_url: user.profile_url,
        // };
        // Add subscription data if available
        // if (userData.subscriptions && userData.subscriptions[0]) {
        //   const s = userData.subscriptions[0];
        //   const PLANS = {
        //     "voice-ai-monthly": "Voice Agent Monthly",
        //     "voice-ai-yearly": "Voice Agent Yearly",
        //     "text-ai-monthly": "Text Agent Monthly",
        //     "text-ai-yearly": "Text Agent Yearly"
        //   };
        //   userData.subscription = {
        //     plan: PLANS[s.planId as keyof typeof PLANS],
        //     status: s.status,
        //     currentPeriodEnd: s.currentPeriodEnd
        //   };
        // }
        // Redirect to frontend with token and user data
        res.redirect(`${process.env.FRONTEND_URL || 'https://voicecon.ai'}/oauth-updated?token=${token}&user=${encodeURIComponent(userId)}`);
    }
    catch (error) {
        console.error('Google OAuth callback error', error);
        res.redirect(`/api/v1/auth/login-failed?error=${encodeURIComponent(error.message)}&stack=${encodeURIComponent(error.stack || 'No stack trace available')}`);
    }
}));
router.get('/login-failed', (req, res) => {
    res.status(400).json({
        message: 'Login failed',
        req: req,
    });
    const errorMessage = req.query.error || 'Unknown error';
    const errorStack = req.query.stack || 'No stack trace available';
    // Log the error details to the console
    console.error(`Login failed: ${errorMessage}`);
    console.error(`Stack trace: ${errorStack}`);
    // Return the error message and stack trace in the response
    res.status(400).json({
        message: 'Login failed',
        error: errorMessage,
        stack: errorStack
    });
});
// Facebook OAuth routes
router.get('/facebook', passport_1.default.authenticate('facebook', {
    scope: ['email']
}));
router.get('/facebook/callback', passport_1.default.authenticate('facebook', {
    session: false,
    failureRedirect: '/login'
}), (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        const userId = user.id.toString();
        const token = generateToken(userId);
        // Format user data similar to regular login
        // const userData = {
        //   id: user.id,
        //   email: user.email,
        //   name: user.name,
        //   profile_url: user.profile_url,
        //   subscriptions: user.subscriptions
        // };
        // Add subscription data if available
        // if (userData.subscriptions && userData.subscriptions[0]) {
        //   const s = userData.subscriptions[0];
        //   const PLANS = {
        //     "voice-ai-monthly": "Voice Agent Monthly",
        //     "voice-ai-yearly": "Voice Agent Yearly",
        //     "text-ai-monthly": "Text Agent Monthly",
        //     "text-ai-yearly": "Text Agent Yearly"
        //   };
        //   userData.subscription = {
        //     plan: PLANS[s.planId as keyof typeof PLANS],
        //     status: s.status,
        //     currentPeriodEnd: s.currentPeriodEnd
        //   };
        // }
        // Redirect to frontend with token and user data
        res.redirect(`${process.env.FRONTEND_URL || 'https://voicecon.ai'}/oauth?token=${token}&user=${encodeURIComponent(userId)}`);
    }
    catch (error) {
        console.error('Facebook OAuth callback error', error);
        res.redirect('/login?error=auth_failed');
    }
}));
router.post('/login', (0, asyncHandler_1.asyncHandler)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { email, password } = req.body;
    try {
        const user = yield User_1.default.findOne({
            where: { email },
            include: [
                { model: Subscription_1.default, as: 'subscriptions' }
            ]
        });
        if (!user || !(yield bcrypt_1.default.compare(password, user.password))) {
            res.status(401).json({ message: 'Invalid credentials' });
            return;
        }
        // Get organization where user is owner
        const organization = yield Organization_1.default.findOne({
            where: { owner_user_id: user.id }
        });
        const token = generateToken(user.id.toString());
        res.status(200).json({
            token,
            user: {
                id: user.id,
                email: user.email,
                name: user.name,
                role: user.role,
                organization: organization
                    ? {
                        organization_id: organization.organization_id,
                        name: organization.name,
                        status: organization.status,
                        subscription_plan_id: organization.subscription_plan_id,
                        custom_features_flags: organization.custom_features_flags
                    }
                    : null,
                subscriptions: user.subscriptions || []
            }
        });
    }
    catch (error) {
        console.error('Login error:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
})));
// Signup route
router.post('/signup', (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { email, password, name } = req.body;
    const hashedPassword = yield bcrypt_1.default.hash(password, 10);
    try {
        const user = yield User_1.default.create({ email, password: hashedPassword, name });
        const token = generateToken(user.id.toString());
        user.password = "";
        res.json({ user, token });
    }
    catch (error) {
        console.error('Signup error', error);
        res.status(500).json({ message: 'Internal server error' });
    }
}));
// Logout route
router.post('/logout', (req, res) => {
    res.json({ message: 'Logged out successfully' });
});
// Route to update profile image
router.put('/update-profile-pic', uploadMiddleware_1.default, (0, asyncHandler_1.asyncHandler)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId } = req.body; // Assume the user ID comes from the logged-in user
    try {
        // Find the user by ID
        const user = yield User_1.default.findByPk(userId);
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }
        // If a file was uploaded, save the URL
        if (req.file) {
            const profileUrl = `/images/profile/${req.file.filename}`;
            user.profile_url = profileUrl;
        }
        // Save the updated user record
        yield user.save();
        // Send the updated user data back
        res.json({ message: 'Profile image updated successfully', user });
    }
    catch (error) {
        console.error('Update profile image error', error);
        res.status(500).json({ message: 'Internal server error' });
    }
})));
// Route to get user by ID
router.get('/user/:id', (0, asyncHandler_1.asyncHandler)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    try {
        const user = yield User_1.default.findByPk(id);
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }
        res.json(user);
    }
    catch (error) {
        console.error('Get user by ID error', error);
        res.status(500).json({ message: 'Internal server error' });
    }
})));
// Route to update user profile
router.put('/update-profile', isAuth_1.default, (0, asyncHandler_1.asyncHandler)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { userId, name, email, company, jobTitle } = req.body;
    console.log(req.body);
    try {
        // Find the user by ID
        const user = yield User_1.default.findByPk(userId);
        console.log(user);
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }
        // Update user fields
        if (name)
            user.name = name;
        if (email)
            user.email = email;
        if (company)
            user.company = company;
        if (jobTitle)
            user.jobTitle = jobTitle;
        // Save the updated user record
        yield user.save();
        // Send the updated user data back
        res.json({
            message: 'Profile updated successfully',
            user
        });
    }
    catch (error) {
        console.error('Update profile error', error);
        res.status(500).json({ message: 'Internal server error' });
    }
})));
exports.default = router;
