"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifySignature = exports.verifyWebhookMiddleware = void 0;
const crypto_1 = __importDefault(require("crypto"));
const WebhookEndpoint_1 = __importDefault(require("../models/WebhookEndpoint"));
const verifyWebhookMiddleware = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const organizationId = req.params.id;
        const signature = req.headers['x-vapi-signature'];
        const timestamp = req.headers['x-vapi-signature'];
        // validation
        if (!signature || timestamp) {
            return res.status(401).json({
                success: false,
                message: "Missing required webhook's headers"
            });
        }
        const currentTimestamp = Math.floor(Date.now() / 1000);
        console.log(currentTimestamp);
        const webhookTimestamp = parseInt(timestamp);
        const timestampTolerance = 300;
        if (Math.abs(currentTimestamp - webhookTimestamp) > timestampTolerance) {
            return res.status(400).json({
                success: false,
                message: "webhook timestamp is too old or too far in the future"
            });
        }
        const webhookEndpoint = yield WebhookEndpoint_1.default.findOne({
            where: {
                organization_id: organizationId,
                status: 'active'
            }
        });
        if (!webhookEndpoint) {
            return res.status(404).json({
                success: false,
                message: 'No active webhook endpoint found for this organization'
            });
        }
        const rawbody = req.rawbody || Buffer.from(JSON.stringify(req.body));
        const bodyString = rawbody.toString('utf-8');
        // verify signature 
        const isValidSignature = (0, exports.verifySignature)(bodyString, signature, timestamp, webhookEndpoint.secret_key);
        if (!isValidSignature) {
            return res.status(401).json({
                success: false,
                message: 'invalid webhook signature'
            });
        }
        yield webhookEndpoint.update({
            last_successful_delivery_at: new Date()
        });
        next();
    }
    catch (error) {
        console.error("Error :" + error);
        return res.status(400).json({
            success: false,
            message: 'webhook verifications failed ',
            error: error instanceof Error ? error.message : 'Unkown error '
        });
    }
});
exports.verifyWebhookMiddleware = verifyWebhookMiddleware;
// signature verification function 
const verifySignature = (payload, signature, timestamp, secretKey) => {
    try {
        // Create the signed payload string
        const signedPayload = `${timestamp}.${payload}`;
        // Calculate expected signature using HMAC SHA256
        const expectedSignature = crypto_1.default
            .createHmac('sha256', secretKey)
            .update(signedPayload, 'utf8')
            .digest('hex');
        // VAPI typically sends signature in format: v1=<signature>
        const formattedExpectedSignature = `v1=${expectedSignature}`;
        // Use timingSafeEqual to prevent timing attacks
        const sigBuffer = Buffer.from(signature, 'utf8');
        const expectedBuffer = Buffer.from(formattedExpectedSignature, 'utf8');
        if (sigBuffer.length !== expectedBuffer.length) {
            return false;
        }
        return crypto_1.default.timingSafeEqual(sigBuffer, expectedBuffer);
    }
    catch (error) {
        console.error('Signature verification error:', error);
        return false;
    }
};
exports.verifySignature = verifySignature;
// // Middleware to capture raw body (add this before JSON parsing)
// export const captureRawBody = (req: WebhookRequest, res: Response, next: NextFunction) => {
//   let data = '';
//   req.setEncoding('utf8');
//   req.on('data', (chunk) => {
//     data += chunk;
//   });
//   req.on('end', () => {
//     req.rawBody = Buffer.from(data, 'utf8');
//     next();
//   });
// };
