"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteCalendarEvent = exports.updateCalendarEvent = exports.fetchCalendarEvents = exports.createCalendarEvent = exports.handleGoogleCalendarCallback = exports.checkConnectedCalendars = exports.connectGoogleCalendar = void 0;
const axios = require('axios');
const dotenv_1 = __importDefault(require("dotenv"));
const CalendarAccount_1 = __importDefault(require("../models/CalendarAccount"));
dotenv_1.default.config();
//  Use AuthenticatedRequest instead of Request
const connectGoogleCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const userId = (_a = req.user) === null || _a === void 0 ? void 0 : _a.id;
    if (!userId) {
        return res.status(401).json({ message: 'User not authenticated' });
    }
    const state = userId.toString();
    console.log('🌐 Sending redirect URI to Google:', process.env.CALENDAR_REDIRECT_URI);
    const redirectUrl = `https://accounts.google.com/o/oauth2/v2/auth?` +
        `client_id=${process.env.GOOGLE_CALENDAR_CLIENT_ID}&` +
        `redirect_uri=${encodeURIComponent(process.env.CALENDAR_REDIRECT_URI)}&` +
        `response_type=code&` +
        `scope=${encodeURIComponent('https://www.googleapis.com/auth/calendar https://www.googleapis.com/auth/userinfo.email')}&` +
        `access_type=offline&` +
        `prompt=consent&` +
        `state=${encodeURIComponent(state)}`;
    // Instead of res.redirect, return the URL to frontend
    return res.status(200).json({ url: redirectUrl });
});
exports.connectGoogleCalendar = connectGoogleCalendar;
// Add this function to your calendarController.js
const checkConnectedCalendars = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const userId = (_a = req.user) === null || _a === void 0 ? void 0 : _a.id;
        if (!userId) {
            res.status(401).json({ message: 'User not authenticated' });
            return;
        }
        // Database se check karo ke user ke kaunse calendars connected hain
        const connectedAccounts = yield CalendarAccount_1.default.findAll({
            where: {
                user_id: userId,
                status: 'active'
            },
            attributes: ['provider', 'email', 'display_name', 'status']
        });
        // Provider names ko array mein convert karo
        const connectedCalendars = connectedAccounts.map(account => {
            // Google provider ko "Google Calendar" name se return karo
            if (account.provider === 'google') {
                return 'Google Calendar';
            }
            // Future ke liye other providers
            return account.provider;
        });
        res.json({
            message: 'Connected calendars fetched successfully',
            connectedCalendars,
            accounts: connectedAccounts.map(account => ({
                provider: account.provider,
                email: account.email,
                displayName: account.display_name,
                status: account.status
            }))
        });
    }
    catch (error) {
        console.error('Error checking connected calendars:', error.message);
        res.status(500).json({
            message: 'Failed to check connected calendars',
            error: error.message
        });
    }
});
exports.checkConnectedCalendars = checkConnectedCalendars;
const handleGoogleCalendarCallback = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const { code, state: userIdString } = req.query;
    if (!code || !userIdString) {
        res.status(400).json({ message: 'Missing code or user ID' });
        return;
    }
    const userId = parseInt(userIdString, 10);
    if (isNaN(userId)) {
        res.status(400).json({ message: 'Invalid user ID format' });
        return;
    }
    try {
        const tokenResponse = yield axios.post('https://oauth2.googleapis.com/token', null, {
            params: {
                code,
                client_id: process.env.GOOGLE_CALENDAR_CLIENT_ID,
                client_secret: process.env.GOOGLE_CALENDAR_CLIENT_SECRET,
                redirect_uri: process.env.CALENDAR_REDIRECT_URI,
                grant_type: 'authorization_code',
            },
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        });
        const { access_token, refresh_token, expires_in } = tokenResponse.data;
        const userInfoResponse = yield axios.get('https://www.googleapis.com/oauth2/v2/userinfo', {
            headers: {
                Authorization: `Bearer ${access_token}`,
            },
        });
        const userInfo = userInfoResponse.data;
        const expiresAt = new Date(Date.now() + expires_in * 1000);
        const existingAccount = yield CalendarAccount_1.default.findOne({
            where: { user_id: userId, provider: 'google' }
        });
        if (existingAccount) {
            //  Use undefined instead of null for optional fields
            yield existingAccount.update({
                email: userInfo.email,
                access_token,
                refresh_token,
                expires_at: expiresAt,
                display_name: userInfo.name,
                status: 'active',
                error_count: 0,
                last_error: undefined, //  Use undefined instead of null
            });
        }
        else {
            yield CalendarAccount_1.default.create({
                user_id: userId,
                provider: 'google',
                email: userInfo.email,
                access_token,
                refresh_token,
                expires_at: expiresAt,
                display_name: userInfo.name,
                calendar_id: 'primary',
                time_zone: 'Asia/Karachi',
                is_primary: true,
                status: 'active',
                error_count: 0,
                // last_error is optional, so we don't need to set it
            });
        }
        res.redirect(`${process.env.FRONTEND_URL || 'http://localhost:5173'}/oauth/calendar?success=true`);
    }
    catch (error) {
        console.error('Google token exchange failed:', ((_a = error === null || error === void 0 ? void 0 : error.response) === null || _a === void 0 ? void 0 : _a.data) || error.message);
        res.status(500).json({ message: 'Token exchange failed', error: error.message });
    }
});
exports.handleGoogleCalendarCallback = handleGoogleCalendarCallback;
// Use AuthenticatedRequest
const createCalendarEvent = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e;
    try {
        const userId = (_a = req.user) === null || _a === void 0 ? void 0 : _a.id;
        const { summary, description, startDateTime, endDateTime, timeZone = 'Asia/Karachi', attendees = [] } = req.body;
        if (!summary || !startDateTime || !endDateTime) {
            res.status(400).json({
                message: 'Missing required fields: summary, startDateTime, endDateTime'
            });
            return;
        }
        const calendarAccount = yield CalendarAccount_1.default.findOne({
            where: { user_id: userId, provider: 'google', status: 'active' }
        });
        if (!calendarAccount) {
            res.status(404).json({
                message: 'Google Calendar not connected. Please connect your calendar first.'
            });
            return;
        }
        const now = new Date();
        if (calendarAccount.expires_at <= now) {
            yield refreshGoogleToken(calendarAccount);
        }
        const eventData = {
            summary,
            description,
            start: {
                dateTime: startDateTime,
                timeZone: calendarAccount.time_zone || timeZone
            },
            end: {
                dateTime: endDateTime,
                timeZone: calendarAccount.time_zone || timeZone
            },
            attendees: attendees.map((email) => ({ email }))
        };
        const response = yield axios.post(`https://www.googleapis.com/calendar/v3/calendars/${calendarAccount.calendar_id}/events`, eventData, {
            headers: {
                Authorization: `Bearer ${calendarAccount.access_token}`,
                'Content-Type': 'application/json'
            }
        });
        const createdEvent = response.data;
        //  Use undefined instead of null
        yield calendarAccount.update({
            error_count: 0,
            last_error: undefined
        });
        res.status(201).json({
            message: 'Calendar event created successfully',
            event: {
                id: createdEvent.id,
                summary: createdEvent.summary,
                description: createdEvent.description,
                startDateTime: createdEvent.start.dateTime,
                endDateTime: createdEvent.end.dateTime,
                htmlLink: createdEvent.htmlLink,
                status: createdEvent.status
            }
        });
    }
    catch (error) {
        console.error('Create calendar event error:', ((_b = error === null || error === void 0 ? void 0 : error.response) === null || _b === void 0 ? void 0 : _b.data) || error.message);
        const userId = (_c = req.user) === null || _c === void 0 ? void 0 : _c.id;
        if (userId) {
            const calendarAccount = yield CalendarAccount_1.default.findOne({
                where: { user_id: userId, provider: 'google' }
            });
            if (calendarAccount) {
                yield calendarAccount.update({
                    error_count: (calendarAccount.error_count || 0) + 1,
                    last_error: error.message
                });
            }
        }
        if (((_d = error === null || error === void 0 ? void 0 : error.response) === null || _d === void 0 ? void 0 : _d.status) === 401) {
            res.status(401).json({
                message: 'Calendar access unauthorized. Please reconnect your calendar.'
            });
        }
        else if (((_e = error === null || error === void 0 ? void 0 : error.response) === null || _e === void 0 ? void 0 : _e.status) === 403) {
            res.status(403).json({
                message: 'Calendar access forbidden. Check permissions.'
            });
        }
        else {
            res.status(500).json({
                message: 'Failed to create calendar event',
                error: error.message
            });
        }
    }
});
exports.createCalendarEvent = createCalendarEvent;
//  Proper type for calendarAccount parameter
const refreshGoogleToken = (calendarAccount) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const response = yield axios.post('https://oauth2.googleapis.com/token', null, {
            params: {
                refresh_token: calendarAccount.refresh_token,
                client_id: process.env.GOOGLE_CALENDAR_CLIENT_ID,
                client_secret: process.env.GOOGLE_CALENDAR_CLIENT_SECRET,
                grant_type: 'refresh_token'
            },
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' }
        });
        const { access_token, expires_in } = response.data;
        const newExpiresAt = new Date(Date.now() + expires_in * 1000);
        yield calendarAccount.update({
            access_token,
            expires_at: newExpiresAt,
            error_count: 0,
            last_error: undefined //  Use undefined instead of null
        });
    }
    catch (error) {
        console.error('Token refresh failed:', ((_a = error === null || error === void 0 ? void 0 : error.response) === null || _a === void 0 ? void 0 : _a.data) || error.message);
        yield calendarAccount.update({
            error_count: (calendarAccount.error_count || 0) + 1,
            last_error: 'Token refresh failed: ' + error.message
        });
        throw new Error('Failed to refresh calendar access token');
    }
});
// Fetch all calendar events
const fetchCalendarEvents = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d;
    try {
        const userId = (_a = req.user) === null || _a === void 0 ? void 0 : _a.id;
        const { startDate, endDate, maxResults = 250 } = req.query;
        if (!userId) {
            res.status(401).json({ message: 'User not authenticated' });
            return;
        }
        const calendarAccount = yield CalendarAccount_1.default.findOne({
            where: { user_id: userId, provider: 'google', status: 'active' }
        });
        if (!calendarAccount) {
            res.status(404).json({
                message: 'Google Calendar not connected. Please connect your calendar first.'
            });
            return;
        }
        // Check if token needs refresh
        const now = new Date();
        if (calendarAccount.expires_at <= now) {
            yield refreshGoogleToken(calendarAccount);
        }
        // Build query parameters
        const params = {
            maxResults,
            singleEvents: true,
            orderBy: 'startTime'
        };
        if (startDate)
            params.timeMin = new Date(startDate).toISOString();
        if (endDate)
            params.timeMax = new Date(endDate).toISOString();
        const response = yield axios.get(`https://www.googleapis.com/calendar/v3/calendars/${calendarAccount.calendar_id}/events`, {
            headers: {
                Authorization: `Bearer ${calendarAccount.access_token}`,
            },
            params
        });
        const events = response.data.items.map((event) => {
            var _a;
            return ({
                id: event.id,
                summary: event.summary,
                description: event.description,
                startDateTime: event.start.dateTime || event.start.date,
                endDateTime: event.end.dateTime || event.end.date,
                location: event.location,
                status: event.status,
                htmlLink: event.htmlLink,
                attendees: ((_a = event.attendees) === null || _a === void 0 ? void 0 : _a.map((attendee) => ({
                    email: attendee.email,
                    responseStatus: attendee.responseStatus
                }))) || []
            });
        });
        res.json({
            message: 'Events fetched successfully',
            events,
            totalCount: response.data.items.length
        });
    }
    catch (error) {
        console.error('Fetch calendar events error:', ((_b = error === null || error === void 0 ? void 0 : error.response) === null || _b === void 0 ? void 0 : _b.data) || error.message);
        if (((_c = error === null || error === void 0 ? void 0 : error.response) === null || _c === void 0 ? void 0 : _c.status) === 401) {
            res.status(401).json({
                message: 'Calendar access unauthorized. Please reconnect your calendar.'
            });
        }
        else if (((_d = error === null || error === void 0 ? void 0 : error.response) === null || _d === void 0 ? void 0 : _d.status) === 403) {
            res.status(403).json({
                message: 'Calendar access forbidden. Check permissions.'
            });
        }
        else {
            res.status(500).json({
                message: 'Failed to fetch calendar events',
                error: error.message
            });
        }
    }
});
exports.fetchCalendarEvents = fetchCalendarEvents;
// Update calendar event
const updateCalendarEvent = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e;
    try {
        const userId = (_a = req.user) === null || _a === void 0 ? void 0 : _a.id;
        const { eventId } = req.params;
        const { summary, description, startDateTime, endDateTime, timeZone = 'Asia/Karachi', attendees = [] } = req.body;
        if (!userId) {
            res.status(401).json({ message: 'User not authenticated' });
            return;
        }
        if (!eventId) {
            res.status(400).json({ message: 'Event ID is required' });
            return;
        }
        const calendarAccount = yield CalendarAccount_1.default.findOne({
            where: { user_id: userId, provider: 'google', status: 'active' }
        });
        if (!calendarAccount) {
            res.status(404).json({
                message: 'Google Calendar not connected. Please connect your calendar first.'
            });
            return;
        }
        // Check if token needs refresh
        const now = new Date();
        if (calendarAccount.expires_at <= now) {
            yield refreshGoogleToken(calendarAccount);
        }
        // Build update data (only include provided fields)
        const updateData = {};
        if (summary)
            updateData.summary = summary;
        if (description)
            updateData.description = description;
        if (startDateTime) {
            updateData.start = {
                dateTime: startDateTime,
                timeZone: calendarAccount.time_zone || timeZone
            };
        }
        if (endDateTime) {
            updateData.end = {
                dateTime: endDateTime,
                timeZone: calendarAccount.time_zone || timeZone
            };
        }
        if (attendees.length > 0) {
            updateData.attendees = attendees.map((email) => ({ email }));
        }
        const response = yield axios.patch(`https://www.googleapis.com/calendar/v3/calendars/${calendarAccount.calendar_id}/events/${eventId}`, updateData, {
            headers: {
                Authorization: `Bearer ${calendarAccount.access_token}`,
                'Content-Type': 'application/json'
            }
        });
        const updatedEvent = response.data;
        res.json({
            message: 'Calendar event updated successfully',
            event: {
                id: updatedEvent.id,
                summary: updatedEvent.summary,
                description: updatedEvent.description,
                startDateTime: updatedEvent.start.dateTime,
                endDateTime: updatedEvent.end.dateTime,
                htmlLink: updatedEvent.htmlLink,
                status: updatedEvent.status
            }
        });
    }
    catch (error) {
        console.error('Update calendar event error:', ((_b = error === null || error === void 0 ? void 0 : error.response) === null || _b === void 0 ? void 0 : _b.data) || error.message);
        if (((_c = error === null || error === void 0 ? void 0 : error.response) === null || _c === void 0 ? void 0 : _c.status) === 401) {
            res.status(401).json({
                message: 'Calendar access unauthorized. Please reconnect your calendar.'
            });
        }
        else if (((_d = error === null || error === void 0 ? void 0 : error.response) === null || _d === void 0 ? void 0 : _d.status) === 404) {
            res.status(404).json({
                message: 'Event not found or already deleted.'
            });
        }
        else if (((_e = error === null || error === void 0 ? void 0 : error.response) === null || _e === void 0 ? void 0 : _e.status) === 403) {
            res.status(403).json({
                message: 'Calendar access forbidden. Check permissions.'
            });
        }
        else {
            res.status(500).json({
                message: 'Failed to update calendar event',
                error: error.message
            });
        }
    }
});
exports.updateCalendarEvent = updateCalendarEvent;
// Delete calendar event
const deleteCalendarEvent = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e;
    try {
        const userId = (_a = req.user) === null || _a === void 0 ? void 0 : _a.id;
        const { eventId } = req.params;
        if (!userId) {
            res.status(401).json({ message: 'User not authenticated' });
            return;
        }
        if (!eventId) {
            res.status(400).json({ message: 'Event ID is required' });
            return;
        }
        const calendarAccount = yield CalendarAccount_1.default.findOne({
            where: { user_id: userId, provider: 'google', status: 'active' }
        });
        if (!calendarAccount) {
            res.status(404).json({
                message: 'Google Calendar not connected. Please connect your calendar first.'
            });
            return;
        }
        // Check if token needs refresh
        const now = new Date();
        if (calendarAccount.expires_at <= now) {
            yield refreshGoogleToken(calendarAccount);
        }
        yield axios.delete(`https://www.googleapis.com/calendar/v3/calendars/${calendarAccount.calendar_id}/events/${eventId}`, {
            headers: {
                Authorization: `Bearer ${calendarAccount.access_token}`,
            }
        });
        res.json({
            message: 'Calendar event deleted successfully',
            eventId
        });
    }
    catch (error) {
        console.error('Delete calendar event error:', ((_b = error === null || error === void 0 ? void 0 : error.response) === null || _b === void 0 ? void 0 : _b.data) || error.message);
        if (((_c = error === null || error === void 0 ? void 0 : error.response) === null || _c === void 0 ? void 0 : _c.status) === 401) {
            res.status(401).json({
                message: 'Calendar access unauthorized. Please reconnect your calendar.'
            });
        }
        else if (((_d = error === null || error === void 0 ? void 0 : error.response) === null || _d === void 0 ? void 0 : _d.status) === 404) {
            res.status(404).json({
                message: 'Event not found or already deleted.'
            });
        }
        else if (((_e = error === null || error === void 0 ? void 0 : error.response) === null || _e === void 0 ? void 0 : _e.status) === 403) {
            res.status(403).json({
                message: 'Calendar access forbidden. Check permissions.'
            });
        }
        else {
            res.status(500).json({
                message: 'Failed to delete calendar event',
                error: error.message
            });
        }
    }
});
exports.deleteCalendarEvent = deleteCalendarEvent;
