"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.bulkUpdateAssistantConfigs = exports.getAssistantConfigsByStatus = exports.updateVapiAssistantId = exports.cloneAssistantConfig = exports.deleteAssistantConfig = exports.updateAssistantConfig = exports.getAssistantConfigById = exports.getAllAssistantConfigs = exports.createAssistantConfig = void 0;
const sequelize_1 = require("sequelize");
const AssistantConfiguration_1 = __importDefault(require("../models/AssistantConfiguration"));
// Error handler function
const handleError = (error) => {
    if (error instanceof Error) {
        return error.message;
    }
    return 'An unknown error occurred';
};
// Create new assistant configuration
const createAssistantConfig = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { name, description, first_message, system_prompt, llm_provider, llm_model, vapi_model_settings, vapi_voice_settings, transcriber_settings, vapi_forwarding_phone_number, server_url_override, server_url_secret_override, metadata } = req.body;
        // Validate required fields
        if (!name || !vapi_model_settings) {
            return res.status(400).json({
                success: false,
                message: 'Name and vapi_model_settings are required'
            });
        }
        const assistantConfig = yield AssistantConfiguration_1.default.create({
            organization_id,
            name,
            description,
            first_message,
            system_prompt,
            llm_provider,
            llm_model,
            vapi_model_settings,
            vapi_voice_settings,
            transcriber_settings,
            vapi_forwarding_phone_number,
            server_url_override,
            server_url_secret_override,
            metadata: metadata || {},
            is_active: true
        });
        return res.status(201).json({
            success: true,
            message: 'Assistant configuration created successfully',
            data: assistantConfig
        });
    }
    catch (error) {
        console.error('Error creating assistant config:', handleError(error));
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            errors: [{ field: 'general', message: handleError(error) }]
        });
    }
});
exports.createAssistantConfig = createAssistantConfig;
// Get all assistant configurations for organization
const getAllAssistantConfigs = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { page = 1, limit = 10, search = '' } = req.query;
        const offset = (Number(page) - 1) * Number(limit);
        const whereClause = {
            organization_id,
            is_active: true
        };
        if (search) {
            whereClause.name = {
                [sequelize_1.Op.iLike]: `%${search}%`
            };
        }
        const { count, rows } = yield AssistantConfiguration_1.default.findAndCountAll({
            where: whereClause,
            limit: Number(limit),
            offset: offset,
            order: [['created_at', 'DESC']]
        });
        const response = {
            data: rows,
            pagination: {
                total: count,
                page: Number(page),
                limit: Number(limit),
                totalPages: Math.ceil(count / Number(limit))
            }
        };
        return res.status(200).json({
            success: true,
            data: response
        });
    }
    catch (error) {
        console.error('Error fetching assistant configs:', handleError(error));
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            errors: [{ field: 'general', message: handleError(error) }]
        });
    }
});
exports.getAllAssistantConfigs = getAllAssistantConfigs;
// Get single assistant configuration by ID
const getAssistantConfigById = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { id } = req.params;
        const assistantConfig = yield AssistantConfiguration_1.default.findOne({
            where: {
                assistant_config_id: id,
                organization_id,
                is_active: true
            }
        });
        if (!assistantConfig) {
            return res.status(404).json({
                success: false,
                message: 'Assistant configuration not found'
            });
        }
        return res.status(200).json({
            success: true,
            data: assistantConfig
        });
    }
    catch (error) {
        console.error('Error fetching assistant config:', handleError(error));
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            errors: [{ field: 'general', message: handleError(error) }]
        });
    }
});
exports.getAssistantConfigById = getAssistantConfigById;
// Update assistant configuration
const updateAssistantConfig = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { id } = req.params;
        const updateData = req.body;
        const assistantConfig = yield AssistantConfiguration_1.default.findOne({
            where: {
                assistant_config_id: id,
                organization_id,
                is_active: true
            }
        });
        if (!assistantConfig) {
            return res.status(404).json({
                success: false,
                message: 'Assistant configuration not found'
            });
        }
        yield assistantConfig.update(Object.assign(Object.assign({}, updateData), { updated_at: new Date() }));
        return res.status(200).json({
            success: true,
            message: 'Assistant configuration updated successfully',
            data: assistantConfig
        });
    }
    catch (error) {
        console.error('Error updating assistant config:', handleError(error));
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            errors: [{ field: 'general', message: handleError(error) }]
        });
    }
});
exports.updateAssistantConfig = updateAssistantConfig;
// Delete assistant configuration (soft delete)
const deleteAssistantConfig = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { id } = req.params;
        const assistantConfig = yield AssistantConfiguration_1.default.findOne({
            where: {
                assistant_config_id: id,
                organization_id,
                is_active: true
            }
        });
        if (!assistantConfig) {
            return res.status(404).json({
                success: false,
                message: 'Assistant configuration not found'
            });
        }
        yield assistantConfig.update({
            is_active: false,
            updated_at: new Date()
        });
        return res.status(200).json({
            success: true,
            message: 'Assistant configuration deleted successfully'
        });
    }
    catch (error) {
        console.error('Error deleting assistant config:', handleError(error));
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            errors: [{ field: 'general', message: handleError(error) }]
        });
    }
});
exports.deleteAssistantConfig = deleteAssistantConfig;
// Clone assistant configuration
const cloneAssistantConfig = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { id } = req.params;
        const { name } = req.body;
        const originalConfig = yield AssistantConfiguration_1.default.findOne({
            where: {
                assistant_config_id: id,
                organization_id,
                is_active: true
            }
        });
        if (!originalConfig) {
            return res.status(404).json({
                success: false,
                message: 'Assistant configuration not found'
            });
        }
        const clonedConfig = yield AssistantConfiguration_1.default.create({
            organization_id,
            name: name || `${originalConfig.name} (Copy)`,
            description: originalConfig.description,
            first_message: originalConfig.first_message,
            system_prompt: originalConfig.system_prompt,
            llm_provider: originalConfig.llm_provider,
            llm_model: originalConfig.llm_model,
            vapi_model_settings: originalConfig.vapi_model_settings,
            vapi_voice_settings: originalConfig.vapi_voice_settings,
            transcriber_settings: originalConfig.transcriber_settings,
            vapi_forwarding_phone_number: originalConfig.vapi_forwarding_phone_number,
            server_url_override: originalConfig.server_url_override,
            server_url_secret_override: originalConfig.server_url_secret_override,
            metadata: originalConfig.metadata,
            is_active: true
        });
        return res.status(201).json({
            success: true,
            message: 'Assistant configuration cloned successfully',
            data: clonedConfig
        });
    }
    catch (error) {
        console.error('Error cloning assistant config:', error);
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
});
exports.cloneAssistantConfig = cloneAssistantConfig;
// Update VAPI assistant ID
const updateVapiAssistantId = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { id } = req.params;
        const { vapi_assistant_id_external } = req.body;
        if (!vapi_assistant_id_external) {
            return res.status(400).json({
                success: false,
                message: 'VAPI assistant ID is required'
            });
        }
        const assistantConfig = yield AssistantConfiguration_1.default.findOne({
            where: {
                assistant_config_id: id,
                organization_id,
                is_active: true
            }
        });
        if (!assistantConfig) {
            return res.status(404).json({
                success: false,
                message: 'Assistant configuration not found'
            });
        }
        yield assistantConfig.update({
            vapi_assistant_id_external,
            updated_at: new Date()
        });
        return res.status(200).json({
            success: true,
            message: 'VAPI assistant ID updated successfully',
            data: assistantConfig
        });
    }
    catch (error) {
        console.error('Error updating VAPI assistant ID:', error);
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
});
exports.updateVapiAssistantId = updateVapiAssistantId;
// Get assistant configurations by status
const getAssistantConfigsByStatus = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { is_active = true } = req.query;
        const assistantConfigs = yield AssistantConfiguration_1.default.findAll({
            where: {
                organization_id,
                is_active: is_active === 'true'
            },
            order: [['created_at', 'DESC']]
        });
        return res.status(200).json({
            success: true,
            data: assistantConfigs
        });
    }
    catch (error) {
        console.error('Error fetching assistant configs by status:', error);
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
});
exports.getAssistantConfigsByStatus = getAssistantConfigsByStatus;
// Bulk update assistant configurations
const bulkUpdateAssistantConfigs = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { organization_id } = req.user;
        const { assistant_config_ids, updates } = req.body;
        if (!assistant_config_ids || !Array.isArray(assistant_config_ids) || assistant_config_ids.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Assistant configuration IDs array is required'
            });
        }
        if (!updates || Object.keys(updates).length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Updates object is required'
            });
        }
        const [updatedCount] = yield AssistantConfiguration_1.default.update(Object.assign(Object.assign({}, updates), { updated_at: new Date() }), {
            where: {
                assistant_config_id: assistant_config_ids,
                organization_id,
                is_active: true
            }
        });
        return res.status(200).json({
            success: true,
            message: `${updatedCount} assistant configurations updated successfully`,
            data: { updatedCount }
        });
    }
    catch (error) {
        console.error('Error bulk updating assistant configs:', error);
        return res.status(500).json({
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
});
exports.bulkUpdateAssistantConfigs = bulkUpdateAssistantConfigs;
